# -*- coding: utf-8 -*-
import os
import re
import time
import json
import threading
import requests
import urllib3
import tkinter as tk
from tkinter import filedialog, messagebox, ttk, scrolledtext, font
from datetime import datetime

# =============================================================================
# ⭐ 基础配置 (Jurisprudence Edition)
# =============================================================================
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

# 请在界面中输入 Key
DEFAULT_API_URL = "https://.........../v1/chat/completions"
DEFAULT_API_KEY = ""
DEFAULT_MODEL_NAME = "......." 
DEFAULT_MAX_TOKENS = 800

# =============================================================================
# 🌍 国际化资源字典 (I18N) - Jurisprudence Edition
# =============================================================================

UI_STRINGS = {
    'zh': {
        'app_title': "BCAITS Juris v1.5 (律制·法学流派模拟评议系统)",
        'group_api': "API 与 模型配置",
        'lbl_url': "API 地址:",
        'lbl_key': "API Key:",
        'lbl_model': "模型名称:",
        'group_file': "律藏输入源 (Vinaya Text) & 分段控制",
        'btn_select_file': "📂 选择文件",
        'lbl_threshold': "分段阈值 (Tokens):",
        # === 新增 UI 文本 ===
        'btn_gen_split': "📄 仅生成分段预览",
        'chk_range': "启用选段处理",
        'lbl_range_from': "从第",
        'lbl_range_to': "段 到第",
        'lbl_range_end': "段",
        # ===================
        'group_lang': "任务矩阵 (法学分析 + 法学流派会诊)", 
        'col_lang': "语言",
        'col_trans': "法条化翻译",
        'col_exe': "法学流派评议",
        'group_log': "系统日志",
        'group_preview': "实时预览",
        'btn_start': "🚀 启动律学解析",
        'btn_stop': "🛑 停止并保存",
        'msg_err_nofile': "错误：请先选择输入文件",
        'msg_err_notask': "错误：请至少勾选一项任务",
        'msg_err_range': "错误：起止段落设置不正确 (必须是数字且 起始<=结束)",
        'msg_stop': "⚠️ 正在停止... (将保存当前进度)",
        'msg_read_start': "读取文本中...",
        'msg_seg_done': "✅ 智能分段完成：共 {} 个片段。",
        'msg_split_only_done': "✅ 分段预览文件已生成！\n请打开查看段号，然后决定处理范围。\n文件位置: {}",
        'msg_split_created': "已生成原始分段文件: {}",
        'msg_file_created': "创建文件: {}",
        'msg_processing': "处理片段 [第 {} 段] ({}/{}): {}",
        'msg_generating': "  -> [{}] 生成 {} ...",
        'msg_done_title': "完成",
        'msg_done_body': "处理完成！\n已生成 [法条译本]、[法学流派评议] 及 [对照版] 文件。",
        'msg_stop_title': "已停止",
        'msg_stop_body': "任务已手动停止。\n\n✅ 截至目前的成果已自动保存。",
        'msg_skip_timeout': "❌ [超时跳过] 第 {} 段处理失败 (2次尝试均超时/错误)，已标记并跳过。",
        'err_fatal': "❌ 错误: {}",
        'lang_zh': "中文", 'lang_en': "英文", 'lang_ja': "日文",
        'lang_ko': "韩文", 'lang_bo': "藏文", 'lang_pali': "巴利文",
        'lang_es': "西班牙语", 'lang_ru': "俄语", 'lang_fr': "法语", 'lang_de': "德语",
        'lang_hi': "印地语", 'lang_no': "挪威语",
    },
    'en': {
        'app_title': "BCAITS Juris v1.5 (Vinaya & Legal Schools Simulation)",
        'group_api': "API & Model Settings",
        'lbl_url': "API Endpoint:",
        'lbl_key': "API Key:",
        'lbl_model': "Model Name:",
        'group_file': "Vinaya Source & Segmentation",
        'btn_select_file': "📂 Browse...",
        'lbl_threshold': "Threshold:",
        # === New UI Text ===
        'btn_gen_split': "📄 Generate Split Only",
        'chk_range': "Process Specific Range",
        'lbl_range_from': "From Seg",
        'lbl_range_to': "To Seg",
        'lbl_range_end': "",
        # ===================
        'group_lang': "Task Matrix (Legal Analysis + School Critique)",
        'col_lang': "Language",
        'col_trans': "Statutory Trans",
        'col_exe': "Legal School Critique",
        'group_log': "Log",
        'group_preview': "Preview",
        'btn_start': "🚀 EXECUTE",
        'btn_stop': "🛑 STOP & SAVE",
        'msg_err_nofile': "Error: Select file.",
        'msg_err_notask': "Error: Select at least one task.",
        'msg_err_range': "Error: Invalid Range (Start <= End).",
        'msg_stop': "⚠️ Stopping... (Progress saved)",
        'msg_read_start': "Reading text...",
        'msg_seg_done': "✅ Segments: {}.",
        'msg_split_only_done': "✅ Split file generated!\nCheck segment numbers to define range.\nFile: {}",
        'msg_split_created': "Created split file: {}",
        'msg_file_created': "Created: {}",
        'msg_processing': "Processing [Seg {}] ({}/{}): {}",
        'msg_generating': "  -> [{}] Generating {} ...",
        'msg_done_title': "Finished",
        'msg_done_body': "Done! Files generated.",
        'msg_stop_title': "Stopped",
        'msg_stop_body': "Task stopped manually. Progress saved.",
        'msg_skip_timeout': "❌ [TIMEOUT SKIP] Segment {} failed after 2 retries.",
        'err_fatal': "❌ Error: {}",
        'lang_zh': "Chinese", 'lang_en': "English", 'lang_ja': "Japanese",
        'lang_ko': "Korean", 'lang_bo': "Tibetan", 'lang_pali': "Pali",
        'lang_es': "Spanish", 'lang_ru': "Russian", 'lang_fr': "French", 'lang_de': "German",
        'lang_hi': "Hindi", 'lang_no': "Norwegian",
    }
}

# =============================================================================
# 🧠 AI 核心配置：法学与社会学 (Jurisprudence & Sociology)
# =============================================================================

class LangConfig:
    @staticmethod
    def get_trans_prompt(lang_code):
        # 翻译 Prompt：法条化、严谨化 (保持不变)
        base = "You are a Legal Historian and Expert in Buddhist Vinaya."
        rules = "Translate the following Vinaya text into formal, statutory legal language (similar to a Civil or Penal Code). Maintain precision. Do not use archaic religious prose; use modern legal terminology."
        
        prompts = {
            'zh': f"{base} 请将以下戒律文本翻译为类似现代法律条文的格式（严谨、书面、法理清晰）。{rules}",
            'en': f"{base} Translate into formal statutory English (Legal English). {rules}",
            'ja': f"{base} 現代の法律条文のような形式で正確に翻訳してください。{rules}",
        }
        return prompts.get(lang_code, prompts.get('en', rules)) + " Keep the original Title."

    @staticmethod
    def get_exe_prompt(lang_code):
        # ⭐ 核心指令修订：匿名化处理 (Anonymous Jurist Simulation)
        CORE_INSTRUCTION = """
        [Role]: You are a Comparative Jurist and Sociologist of Religion.
        [Objective]: Analyze the Vinaya text not just as religious rules, but as an ancient legal and social system.
        
        [Analysis Framework]:
        1. **Juridical Analysis (法理构成)**:
           - Subject (Who), Object (What interest is protected?), Act (Actus Reus), Intent (Mens Rea).
           - Exceptions/Exemptions (if any).
        
        2. **Sociological Origins (社会学溯源)**:
           - Why was this rule created *at that time*? (e.g., economic scarcity, public reputation).
        
        3. **Simulation of 5 Legal Perspectives (五大法学视角模拟会诊)**:
           **INSTRUCTION**: Select 5 distinct legal perspectives from the list below to critique this Vinaya rule.
           *Important*: Do NOT use specific names of modern jurists in your output. Use the titles provided below.
           
           *Candidates*:
           - **Expert on Law as Integrity (法律整全性专家)**: Critique based on moral principles, consistency, and rights (Interpretative approach).
           - **Expert on Law and Economics (法律经济学专家)**: Critique based on efficiency, wealth maximization, and cost-benefit analysis.
           - **Common Law Traditionalist (e.g., Blackstone/Coke)**: Critique from the tradition of precedent and reasonable man.
           - **Roman Law Jurist (e.g., Gaius/Ulpian)**: Critique from Civil Law categories (Personae, Res, Actiones).
           - **Constitutionalist (e.g., Jefferson/Madison)**: Critique based on liberty, self-governance, and constitutional rights.
           - **Legal Positivist (e.g., Hart)**: Critique based on the separation of law and morals, primary vs secondary rules.
           
           *Requirement*: Each expert must offer a sharp, distinct comment (1-2 sentences each) reflecting their specific school of thought.

        4. **Modern Review & Evolution (现代审视)**:
           - Does this rule conflict with modern Human Rights or Gender Equality?
           - Explicitly point out obsolete parts.
        """
        
        prompts = {
            'zh': f"""{CORE_INSTRUCTION}
            输出格式：
            【法条重述】(用现代法律语言概括核心戒条)
            【法理分析】(构成要件：主体/客体/主观/客观)
            【社会学背景】(制定缘起)
            【五大法学视角会诊】(请使用“法律整全性专家”、“法律经济学专家”等头衔进行评议，模拟其学派口吻)
            【现代文明审视】(批判性回顾与演化方向)""",
            
            'en': f"""{CORE_INSTRUCTION}
            Output Format:
            [Legal Restatement]
            [Juridical Analysis]
            [Sociological Context]
            [The Council of 5 Legal Perspectives] (Use titles like "Expert on Law as Integrity", "Expert on Law and Economics", etc.)
            [Modern Critical Review]""",
        }
        return prompts.get(lang_code, prompts.get('en', CORE_INSTRUCTION))

    @staticmethod
    def get_file_suffix(lang_code, task_type):
        file_lang_code = 'cn' if lang_code == 'zh' else lang_code
        names = {
            'trans': {'zh': '法条译本', 'en': 'Statutory_Trans', 'ja': '法文翻訳'},
            'exe': {'zh': '法学流派评议', 'en': 'Legal_School_Critique', 'ja': '法学派評論'},
            'combined': {'zh': '律学对照版', 'en': 'Vinaya_Study', 'ja': '律学対照'}
        }
        func_name = names.get(task_type, {}).get(lang_code, 'Output')
        return f"_{func_name}.{file_lang_code}.txt"

    @staticmethod
    def validate_pali(text):
        clean_text = re.sub(r'[（(][^)）]*[)）]', '', text.replace('\ufeff', ''))
        if re.search(r'[\u0900-\u097F]', clean_text): return False, "Detected Devanagari"
        return True, None

# =============================================================================
# 📜 智能分段
# =============================================================================

class ZenTextProcessor:
    def __init__(self):
        self.header_pattern = re.compile(r'^\s*([^\s]+(?:戒|篇|聚|犍度|法|律|卷第.+)|Rule\s+\d+|Article\s+\d+)\s*$')
        self.sentence_end_pattern = re.compile(r'([。！？；.!?;།༎।]+)')

    def preprocess_text(self, full_text):
        if len(full_text) > 1000 and full_text.count('\n') < 10:
            full_text = self.sentence_end_pattern.sub(r'\1\n', full_text)
        return full_text

    def smart_segmentation(self, full_text, max_chars=3000):
        full_text = self.preprocess_text(full_text)
        lines = full_text.split('\n')
        segments = []
        current_title = "Excerpt / 选段"
        current_buffer = []
        current_count = 0
        is_standard_style = (sum(1 for line in lines[:100] if self.header_pattern.match(line)) > 2)

        for line in lines:
            line = line.rstrip()
            if not line: continue
            should_split = False; new_title = None
            is_header = False
            
            if is_standard_style:
                if self.header_pattern.match(line) and len(line) < 40: is_header = True
            else:
                if len(line) < 40 and not self.sentence_end_pattern.search(line[-1:]):
                    if current_count > 100: is_header = True
            
            if is_header: should_split = True; new_title = line.strip()
            if current_count + len(line) > max_chars: should_split = True; new_title = new_title if new_title else current_title + " (Part II)"

            if should_split and current_buffer:
                segments.append({"title": current_title, "content": "\n".join(current_buffer)})
                if new_title:
                    current_title = new_title
                    if is_header: current_buffer = []; current_count = 0
                    else: current_buffer = [line]; current_count = len(line)
                else: current_buffer = [line]; current_count = len(line)
            else:
                if is_header: current_title = line.strip()
                else: current_buffer.append(line); current_count += len(line)
        
        if current_buffer: segments.append({"title": current_title, "content": "\n".join(current_buffer)})
        return segments

# =============================================================================
# 🤖 AI 引擎 (超时重试机制)
# =============================================================================

class AiEngine:
    def __init__(self, api_url, api_key):
        self.api_url = api_url
        self.api_key = api_key
    
    def process(self, title, content, system_prompt, model_name, validator=None):
        user_prompt = f"Target Text: {title}\n\nContent:\n{content}"
        messages = [{"role": "system", "content": system_prompt}, {"role": "user", "content": user_prompt}]
        headers = {"Authorization": f"Bearer {self.api_key}", "Content-Type": "application/json"}
        
        # ⭐ 容错升级：5分钟超时，2次重试
        MAX_RETRIES = 2
        TIMEOUT_SECONDS = 300 
        
        for attempt in range(MAX_RETRIES):
            try:
                # 保持 0.4 中温
                payload = {"model": model_name, "messages": messages, "temperature": 0.4, "max_tokens": 4096}
                resp = requests.post(self.api_url, headers=headers, json=payload, timeout=TIMEOUT_SECONDS)
                
                if resp.status_code != 200: 
                    print(f"API Error {resp.status_code}: {resp.text}")
                    time.sleep(2)
                    continue

                res = resp.json()['choices'][0]['message']['content']
                if validator:
                    if not validator(res)[0]: continue
                return res
            except requests.exceptions.Timeout:
                print(f"Timeout (Attempt {attempt+1}/{MAX_RETRIES})")
            except Exception as e:
                print(f"Exception (Attempt {attempt+1}/{MAX_RETRIES}): {str(e)}")
            time.sleep(3)
        
        return "[FAILED: TIMEOUT OR ERROR]"

# =============================================================================
# 🚀 启动器
# =============================================================================

class LanguageLauncher:
    def __init__(self):
        self.root = tk.Tk(); self.root.title("Language Selection"); 
        w,h=550,550; x,y=(self.root.winfo_screenwidth()-w)//2, (self.root.winfo_screenheight()-h)//2
        self.root.geometry(f"{w}x{h}+{x}+{y}"); self.selected_lang=None
        ttk.Label(self.root, text="Select Interface Language\n请选择界面语言", font=("Arial",14), justify=tk.CENTER).pack(pady=20)
        f=ttk.Frame(self.root); f.pack(pady=10)
        langs=[
            ("中文 (Chinese)",'zh'),("English",'en'),
            ("日本語 (Japanese)",'ja'),("한국어 (Korean)",'ko'),
            ("བོད་ཡིག (Tibetan)",'bo'),("Pāḷi (Roman)",'pali'),
            ("Español (Spanish)",'es'),("Русский (Russian)",'ru'),
            ("Français (French)",'fr'),("Deutsch (German)",'de'),
            ("हिन्दी (Hindi)",'hi'),("Norsk (Norwegian)",'no')
        ]
        for i,(n,c) in enumerate(langs): 
            ttk.Button(f,text=n,command=lambda x=c:self.sel(x),width=22).grid(row=i//2,column=i%2,padx=10,pady=10)
    def sel(self,c): self.selected_lang=c; self.root.destroy()
    def run(self): self.root.mainloop(); return self.selected_lang

# =============================================================================
# 🖥️ GUI 主程序
# =============================================================================

class ZenUniversalApp:
    def __init__(self, root, ui_lang='zh'):
        self.root = root; self.ui_lang = ui_lang
        self.T = UI_STRINGS.get(ui_lang, UI_STRINGS['en'])
        self.root.title(self.T['app_title'])
        self.root.geometry("1200x950")
        self.processor = ZenTextProcessor()
        self.vars_trans = {}; self.vars_exe = {}
        self.is_running = False; self.stop_event = threading.Event()
        self._setup_ui()

    def _setup_ui(self):
        top_container = ttk.Frame(self.root, padding=10); top_container.pack(fill=tk.X)
        api_group = ttk.LabelFrame(top_container, text=self.T['group_api'], padding=10)
        api_group.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        f1 = ttk.Frame(api_group); f1.pack(fill=tk.X, pady=2)
        ttk.Label(f1, text=self.T['lbl_url'], width=10).pack(side=tk.LEFT)
        self.api_url = tk.StringVar(value=DEFAULT_API_URL); ttk.Entry(f1, textvariable=self.api_url).pack(side=tk.LEFT, fill=tk.X, expand=True, padx=5)
        ttk.Label(f1, text=self.T['lbl_key'], width=8).pack(side=tk.LEFT)
        self.api_key = tk.StringVar(value=DEFAULT_API_KEY); ttk.Entry(f1, textvariable=self.api_key, show="*", width=20).pack(side=tk.LEFT, padx=5)
        f2 = ttk.Frame(api_group); f2.pack(fill=tk.X, pady=5)
        ttk.Label(f2, text=self.T['lbl_model'], width=10).pack(side=tk.LEFT)
        self.model_name = tk.StringVar(value=DEFAULT_MODEL_NAME); ttk.Entry(f2, textvariable=self.model_name).pack(side=tk.LEFT, fill=tk.X, expand=True, padx=5)
        
        action_frame = ttk.Frame(top_container, padding=(10, 0, 0, 0)); action_frame.pack(side=tk.RIGHT, fill=tk.Y)
        style = ttk.Style(); style.configure("Big.TButton", font=("Arial", 12, "bold")); style.configure("Stop.TButton", font=("Arial", 10))
        self.btn_start = ttk.Button(action_frame, text=self.T['btn_start'], command=self.start, style="Big.TButton", width=15)
        self.btn_start.pack(side=tk.TOP, fill=tk.BOTH, expand=True, pady=(5, 5), ipady=10)
        self.btn_stop = ttk.Button(action_frame, text=self.T['btn_stop'], command=self.stop, style="Stop.TButton")
        self.btn_stop.pack(side=tk.BOTTOM, fill=tk.X, pady=(0, 5))

        mid_frame = ttk.Frame(self.root, padding=10); mid_frame.pack(fill=tk.X)
        file_group = ttk.LabelFrame(mid_frame, text=self.T['group_file'], padding=10)
        file_group.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=(0, 5))
        
        # 文件选择
        f_box = ttk.Frame(file_group); f_box.pack(fill=tk.X)
        self.file_path = tk.StringVar(); ttk.Entry(f_box, textvariable=self.file_path).pack(side=tk.LEFT, fill=tk.X, expand=True)
        ttk.Button(f_box, text=self.T['btn_select_file'], command=self._select_file).pack(side=tk.LEFT, padx=5)
        
        # 分段阈值 + 仅生成分段按钮
        t_box = ttk.Frame(file_group); t_box.pack(fill=tk.X, pady=5)
        ttk.Label(t_box, text=self.T['lbl_threshold']).pack(side=tk.LEFT)
        self.ts_limit = tk.IntVar(value=DEFAULT_MAX_TOKENS); ttk.Entry(t_box, textvariable=self.ts_limit, width=8).pack(side=tk.LEFT, padx=5)
        ttk.Button(t_box, text=self.T['btn_gen_split'], command=self.generate_split_only).pack(side=tk.LEFT, padx=20) 

        # 选段处理区域
        r_box = ttk.Frame(file_group); r_box.pack(fill=tk.X, pady=5)
        self.var_use_range = tk.BooleanVar(value=False)
        self.chk_range = ttk.Checkbutton(r_box, text=self.T['chk_range'], variable=self.var_use_range, command=self._toggle_range_inputs)
        self.chk_range.pack(side=tk.LEFT)
        
        ttk.Label(r_box, text=" |  " + self.T['lbl_range_from']).pack(side=tk.LEFT, padx=(10,0))
        self.var_range_start = tk.IntVar(value=1)
        self.ent_start = ttk.Entry(r_box, textvariable=self.var_range_start, width=5, state='disabled')
        self.ent_start.pack(side=tk.LEFT, padx=5)
        
        ttk.Label(r_box, text=self.T['lbl_range_to']).pack(side=tk.LEFT)
        self.var_range_end = tk.IntVar(value=10)
        self.ent_end = ttk.Entry(r_box, textvariable=self.var_range_end, width=5, state='disabled')
        self.ent_end.pack(side=tk.LEFT, padx=5)
        ttk.Label(r_box, text=self.T['lbl_range_end']).pack(side=tk.LEFT)

        lang_group = ttk.LabelFrame(mid_frame, text=self.T['group_lang'], padding=10)
        lang_group.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=(5, 0))
        ttk.Label(lang_group, text=self.T['col_lang'], font=("Arial", 9, "bold")).grid(row=0, column=0, sticky=tk.W)
        ttk.Label(lang_group, text=self.T['col_trans'], font=("Arial", 9, "bold")).grid(row=0, column=1)
        ttk.Label(lang_group, text=self.T['col_exe'], font=("Arial", 9, "bold")).grid(row=0, column=2)
        
        # 12 种语言
        order = ['zh', 'en', 'ja', 'ko', 'bo', 'pali', 'es', 'ru', 'fr', 'de', 'hi', 'no']
        for i, key in enumerate(order):
            row = i + 1
            ttk.Label(lang_group, text=self.T.get(f'lang_{key}', key)).grid(row=row, column=0, sticky=tk.W, padx=5, pady=2)
            val_trans = False; val_exe = (key == self.ui_lang) 
            vt = tk.BooleanVar(value=val_trans); self.vars_trans[key] = vt
            ve = tk.BooleanVar(value=val_exe); self.vars_exe[key] = ve
            ttk.Checkbutton(lang_group, variable=vt).grid(row=row, column=1)
            ttk.Checkbutton(lang_group, variable=ve).grid(row=row, column=2)

        main_content = ttk.PanedWindow(self.root, orient=tk.VERTICAL); main_content.pack(fill=tk.BOTH, expand=True, padx=10, pady=(0, 10))
        log_frame = ttk.LabelFrame(main_content, text=self.T['group_log']); main_content.add(log_frame, weight=1)
        self.log_text = scrolledtext.ScrolledText(log_frame, height=8, font=("Consolas", 9), state='normal'); self.log_text.pack(fill=tk.BOTH, expand=True)
        preview_frame = ttk.LabelFrame(main_content, text=self.T['group_preview']); main_content.add(preview_frame, weight=4)
        self.preview_area = scrolledtext.ScrolledText(preview_frame, font=("微软雅黑", 11)); self.preview_area.pack(fill=tk.BOTH, expand=True)
        self.progress = ttk.Progressbar(self.root, mode='determinate'); self.progress.pack(fill=tk.X, padx=10, pady=(0, 5))

    def _select_file(self):
        f = filedialog.askopenfilename(filetypes=[("Text Files", "*.txt")])
        if f: self.file_path.set(f)
    def log(self, msg):
        self.log_text.insert(tk.END, f"[{datetime.now().strftime('%H:%M:%S')}] {msg}\n"); self.log_text.see(tk.END)
    
    def _toggle_range_inputs(self):
        st = 'normal' if self.var_use_range.get() else 'disabled'
        self.ent_start.config(state=st); self.ent_end.config(state=st)

    # ⭐ 仅生成分段
    def generate_split_only(self):
        if not self.file_path.get(): messagebox.showerror("Error", self.T['msg_err_nofile']); return
        try:
            input_file = self.file_path.get(); base_name = os.path.splitext(input_file)[0]
            with open(input_file, 'r', encoding='utf-8') as f: content = f.read()
            segments = self.processor.smart_segmentation(content, self.ts_limit.get())
            split_file_path = f"{base_name}_split.txt"
            with open(split_file_path, 'w', encoding='utf-8') as f:
                for i, seg in enumerate(segments):
                    f.write(f"【第 {i+1} 段】 {seg['title']}\n{seg['content']}\n\n{'='*40}\n\n")
            self.log(self.T['msg_split_only_done'].format(os.path.basename(split_file_path)))
            messagebox.showinfo("Done", self.T['msg_split_only_done'].format(os.path.basename(split_file_path)))
        except Exception as e:
            messagebox.showerror("Error", str(e))

    def stop(self):
        if self.is_running: self.stop_event.set(); self.log(self.T['msg_stop'])
    
    def start(self):
        if not self.file_path.get(): messagebox.showerror("Error", self.T['msg_err_nofile']); return
        
        # 检查选段
        r_start, r_end = None, None
        if self.var_use_range.get():
            try:
                s = self.var_range_start.get(); e = self.var_range_end.get()
                if s <= 0 or e < s: raise ValueError
                r_start, r_end = s, e
            except: messagebox.showerror("Error", self.T['msg_err_range']); return

        selected_tasks = []
        selected_langs = set()
        for lang in ['zh', 'en', 'ja', 'ko', 'bo', 'pali', 'es', 'ru', 'fr', 'de', 'hi', 'no']:
            if self.vars_trans[lang].get(): selected_tasks.append({'lang': lang, 'type': 'trans'}); selected_langs.add(lang)
            if self.vars_exe[lang].get(): selected_tasks.append({'lang': lang, 'type': 'exe'}); selected_langs.add(lang)
        if not selected_tasks: messagebox.showerror("Error", self.T['msg_err_notask']); return
        
        self.is_running = True; self.stop_event.clear(); self.btn_start.config(state=tk.DISABLED)
        # 传入范围参数
        threading.Thread(target=self._run_process, args=(selected_tasks, list(selected_langs), r_start, r_end), daemon=True).start()
    
    def _run_process(self, tasks, active_langs, r_start, r_end):
        input_file = self.file_path.get(); base_name = os.path.splitext(input_file)[0]
        stopped_by_user = False
        try:
            self.log(self.T['msg_read_start'])
            with open(input_file, 'r', encoding='utf-8') as f: content = f.read()
            segments = self.processor.smart_segmentation(content, self.ts_limit.get()); total_segs = len(segments)
            self.log(self.T['msg_seg_done'].format(total_segs))
            
            # 生成分段文件
            split_file_path = f"{base_name}_split.txt"
            if not os.path.exists(split_file_path):
                with open(split_file_path, 'w', encoding='utf-8') as f:
                    for i, seg in enumerate(segments): f.write(f"【第 {i+1} 段】 {seg['title']}\n{seg['content']}\n\n{'='*40}\n\n")
                self.log(self.T['msg_split_created'].format(os.path.basename(split_file_path)))

            handles = {}
            # 追加模式 'a'
            for t in tasks:
                suffix = LangConfig.get_file_suffix(t['lang'], t['type']); out_path = base_name + suffix
                f = open(out_path, 'a', encoding='utf-8')
                handles[(t['lang'], t['type'])] = f
            for lang in active_langs:
                suffix = LangConfig.get_file_suffix(lang, 'combined'); out_path = base_name + suffix
                f = open(out_path, 'a', encoding='utf-8')
                handles[(lang, 'combined')] = f

            ai = AiEngine(self.api_url.get(), self.api_key.get())
            for i, seg in enumerate(segments):
                seg_num = i + 1
                
                # ⭐ 范围跳过
                if r_start and r_end:
                    if seg_num < r_start: continue
                    if seg_num > r_end: break

                if self.stop_event.is_set(): stopped_by_user = True; break
                
                title, text = seg['title'], seg['content']
                self.log(self.T['msg_processing'].format(seg_num, seg_num, total_segs, title))
                segment_results = {lang: {'trans': None, 'exe': None} for lang in active_langs}

                for t in tasks:
                    if self.stop_event.is_set(): stopped_by_user = True; break
                    lang, type_ = t['lang'], t['type']
                    prompt = LangConfig.get_trans_prompt(lang) if type_ == 'trans' else LangConfig.get_exe_prompt(lang)
                    validator = LangConfig.validate_pali if lang == 'pali' else None
                    
                    lang_name = self.T.get(f'lang_{lang}', lang); task_display = self.T['col_trans'] if type_ == 'trans' else self.T['col_exe']
                    self.log(self.T['msg_generating'].format(lang_name, task_display))
                    
                    result = ai.process(title, text, prompt, self.model_name.get(), validator)
                    
                    # ⭐ 失败跳过
                    if "[FAILED" in result:
                        self.log(self.T['msg_skip_timeout'].format(seg_num))
                        result = f"⚠️ [FAILED] Segment {seg_num} skipped due to API Timeout/Error."

                    # 写入结果
                    handles[(lang, type_)].write(f"【第 {seg_num} 段 - {title}】\n{result}\n\n{'='*60}\n\n"); handles[(lang, type_)].flush()
                    segment_results[lang][type_] = result
                    
                    self.preview_area.delete(1.0, tk.END)
                    self.preview_area.insert(tk.END, f"--- {lang_name} [{type_}] ---\n{title}\n\n{result}")
                
                if stopped_by_user: break

                # 写入对照版
                for lang in active_langs:
                    combined_f = handles[(lang, 'combined')]
                    combined_text = f"【第 {seg_num} 段 - {title}】\n\n--- Source ---\n{text}\n"
                    if segment_results[lang].get('trans'): combined_text += f"\n--- Statutory Trans ---\n{segment_results[lang]['trans']}\n"
                    if segment_results[lang].get('exe'): combined_text += f"\n--- Jurist Critique ---\n{segment_results[lang]['exe']}\n"
                    combined_text += f"\n{'='*60}\n\n"
                    combined_f.write(combined_text); combined_f.flush()

                self.progress['value'] = (seg_num / total_segs) * 100
            
            if stopped_by_user:
                self.log("🛑 STOPPED.")
                messagebox.showinfo(self.T.get('msg_stop_title', "已停止"), self.T.get('msg_stop_body', "任务已停止。\n进度已保存。"))
            else:
                self.log("DONE!")
                messagebox.showinfo(self.T['msg_done_title'], self.T['msg_done_body'])

        except Exception as e:
            self.log(self.T['err_fatal'].format(str(e))); messagebox.showerror("Error", str(e))
        finally:
            if 'handles' in locals():
                for f in handles.values(): f.flush(); f.close()
            self.is_running = False; self.btn_start.config(state=tk.NORMAL); self.progress['value'] = 0

if __name__ == "__main__":
    launcher = LanguageLauncher()
    selected_lang = launcher.run()
    if selected_lang:
        root = tk.Tk()
        app = ZenUniversalApp(root, ui_lang=selected_lang)
        root.mainloop()
